// script.js

const taskForm = document.getElementById("taskForm");
const taskIdInput = document.getElementById("taskId");
const titleInput = document.getElementById("title");
const descriptionInput = document.getElementById("description");
const statusSelect = document.getElementById("status");
const dueDateInput = document.getElementById("dueDate");
const submitBtn = document.getElementById("submitBtn");
const cancelEditBtn = document.getElementById("cancelEditBtn");
const messageBox = document.getElementById("messageBox");
const formTitle = document.getElementById("formTitle");
const taskList = document.getElementById("taskList");

// Show messages
function showMessage(text, type = "success") {
    messageBox.textContent = text;
    messageBox.className = type; // CSS will style based on this class
}

// Clear messages
function clearMessage() {
    messageBox.textContent = "";
    messageBox.className = "";
}

// Client-side validation
function validateForm() {
    const title = titleInput.value.trim();
    const description = descriptionInput.value.trim();
    const dueDate = dueDateInput.value;

    if (!title || !description) {
        showMessage("Title and description cannot be empty.", "error");
        return false;
    }

    // Optional date validation (YYYY-MM-DD - but browser already does basic validation)
    if (dueDate && !/^\d{4}-\d{2}-\d{2}$/.test(dueDate)) {
        showMessage("Please enter a valid due date.", "error");
        return false;
    }

    return true;
}

// Reset form to initial state (add mode)
function resetForm() {
    taskIdInput.value = "";
    titleInput.value = "";
    descriptionInput.value = "";
    statusSelect.value = "pending";
    dueDateInput.value = "";

    submitBtn.textContent = "Add Task";
    formTitle.textContent = "Add New Task";
    cancelEditBtn.style.display = "none";
}

// Populate form with existing data (edit mode)
function fillFormForEdit(task) {
    taskIdInput.value = task.id;
    titleInput.value = task.title;
    descriptionInput.value = task.description;
    statusSelect.value = task.status;
    dueDateInput.value = task.dueDate || "";

    submitBtn.textContent = "Update Task";
    formTitle.textContent = "Edit Task";
    cancelEditBtn.style.display = "inline-block";
}

// Load all tasks from backend
function loadTasks() {
    fetch("/tasks")
        .then(res => res.json())
        .then(tasks => {
            taskList.innerHTML = "";

            if (tasks.length === 0) {
                taskList.innerHTML = "<p>No tasks yet.</p>";
                return;
            }

            tasks.forEach(task => {
                const div = document.createElement("div");
                div.className = "task";

                div.innerHTML = `
                    <h3>${task.title}</h3>
                    <p>${task.description}</p>
                    <p><strong>Status:</strong> ${task.status}</p>
                    <p><strong>Due:</strong> ${task.dueDate || "N/A"}</p>
                    <button class="editBtn">Edit</button>
                    <button class="deleteBtn">Delete</button>
                `;

                const editBtn = div.querySelector(".editBtn");
                const deleteBtn = div.querySelector(".deleteBtn");

                editBtn.addEventListener("click", () => {
                    fillFormForEdit(task);
                    clearMessage();
                });

                deleteBtn.addEventListener("click", () => {
                    deleteTask(task.id);
                });

                taskList.appendChild(div);
            });
        })
        .catch(err => {
            console.error(err);
            showMessage("Error loading tasks.", "error");
        });
}

// Handle form submit (Add or Update)
taskForm.addEventListener("submit", event => {
    event.preventDefault();
    clearMessage();

    if (!validateForm()) return;

    const id = taskIdInput.value;
    const data = {
        title: titleInput.value.trim(),
        description: descriptionInput.value.trim(),
        status: statusSelect.value,
        dueDate: dueDateInput.value
    };

    // If there is an id, we're updating; else, we're creating
    if (id) {
        // UPDATE (PUT /tasks/:id)
        fetch(`/tasks/${id}`, {
            method: "PUT",
            headers: { "Content-Type": "application/json" },
            body: JSON.stringify(data)
        })
            .then(res => res.json())
            .then(result => {
                if (result.message) {
                    showMessage(result.message, "success");
                }
                resetForm();
                loadTasks();
            })
            .catch(err => {
                console.error(err);
                showMessage("Error updating task.", "error");
            });
    } else {
        // CREATE (POST /tasks)
        fetch("/tasks", {
            method: "POST",
            headers: { "Content-Type": "application/json" },
            body: JSON.stringify(data)
        })
            .then(res => res.json())
            .then(result => {
                if (result.message) {
                    showMessage(result.message, "success");
                }
                resetForm();
                loadTasks();
            })
            .catch(err => {
                console.error(err);
                showMessage("Error creating task.", "error");
            });
    }
});

// Cancel edit button
cancelEditBtn.addEventListener("click", () => {
    resetForm();
    clearMessage();
});

// Delete a task
function deleteTask(id) {
    if (!confirm("Are you sure you want to delete this task?")) return;

    fetch(`/tasks/${id}`, {
        method: "DELETE"
    })
        .then(res => res.json())
        .then(result => {
            if (result.message) {
                showMessage(result.message, "success");
            }
            loadTasks();
        })
        .catch(err => {
            console.error(err);
            showMessage("Error deleting task.", "error");
        });
}

// Initial load
loadTasks();
resetForm();
